/***********************************************************************************************************************
 * File Name    : uart.c
 * Description  : Contains UART functions definition.
 **********************************************************************************************************************/
/***********************************************************************************************************************
* Copyright (c) 2020 - 2024 Renesas Electronics Corporation and/or its affiliates
*
* SPDX-License-Identifier: BSD-3-Clause
***********************************************************************************************************************/
#include "uart_ep.h"

#include "FreeRTOS.h"
#include "task.h"
#include "FreeRTOS_CLI.h"
#include "cli_app.h"

extern TaskHandle_t cli_command_task;
/*******************************************************************************************************************//**
 * @addtogroup r_sci_uart_ep
 * @{
 **********************************************************************************************************************/

/*
 * Private function declarations
 */

/*
 * Private global variables
 */
/* Temporary buffer to save data from receive buffer for further processing */
static uint8_t g_temp_buffer[DATA_LENGTH] = {RESET_VALUE};

/* Counter to update g_temp_buffer index */
static volatile uint8_t g_counter_var = RESET_VALUE;

/* Flag to check whether data is received or not */
static volatile uint8_t g_data_received_flag = false;

/* Flag for user callback */
static volatile uint8_t g_uart_event = RESET_VALUE;

//uart tx empty
static volatile bool uart_send_complete_flag = false;

uint32_t cli_recv;


/*******************************************************************************************************************//**
 * @brief       Initialize  UART.
 * @param[in]   None
 * @retval      FSP_SUCCESS         Upon successful open and start of timer
 * @retval      Any Other Error code apart from FSP_SUCCESS  Unsuccessful open
 ***********************************************************************************************************************/
fsp_err_t uart_initialize(void)
{
    fsp_err_t err = FSP_SUCCESS;

    /* Initialize UART channel with baud rate 115200 */
#if (BSP_FEATURE_SCI_VERSION == 2U)
    err = R_SCI_B_UART_Open (&g_uart9_ctrl, &g_uart9_cfg);
#else
    err = R_SCI_UART_Open (&g_uart9_ctrl, &g_uart9_cfg);
#endif
    if (FSP_SUCCESS != err)
    {
        
    }
    return err;
}

/*****************************************************************************************************************
 *  @brief       print user message to terminal
 *  @param[in]   p_msg
 *  @retval      FSP_SUCCESS                Upon success
 *  @retval      FSP_ERR_TRANSFER_ABORTED   Upon event failure
 *  @retval      Any Other Error code apart from FSP_SUCCESS,  Unsuccessful write operation
 ****************************************************************************************************************/
fsp_err_t uart_print_user_msg(uint8_t *p_msg)
{
    fsp_err_t err   = FSP_SUCCESS;
    uint8_t msg_len = RESET_VALUE;
    uint32_t local_timeout = (DATA_LENGTH * UINT16_MAX);
    uint8_t *p_temp_ptr = (uint8_t *)p_msg;

    /* Calculate length of message received */
    msg_len = ((uint8_t)(strlen((char *)p_temp_ptr)));

    /* Reset callback capture variable */
    g_uart_event = RESET_VALUE;

    /* Writing to terminal */
#if (BSP_FEATURE_SCI_VERSION == 2U)
    err = R_SCI_B_UART_Write (&g_uart_ctrl, p_msg, msg_len);
#else
    err = R_SCI_UART_Write (&g_uart9_ctrl, p_msg, msg_len);
#endif

    if (FSP_SUCCESS != err)
    {
        
        return err;
    }

    /* Check for event transfer complete */
    while ((UART_EVENT_TX_COMPLETE != g_uart_event) && (--local_timeout))
    {
        /* Check if any error event occurred */
        if (UART_ERROR_EVENTS == g_uart_event)
        {
            return FSP_ERR_TRANSFER_ABORTED;
        }
    }
    if(RESET_VALUE == local_timeout)
    {
        err = FSP_ERR_TIMEOUT;
    }
    return err;
}

/*******************************************************************************************************************//**
 *  @brief       Deinitialize SCI UART module
 *  @param[in]   None
 *  @retval      None
 **********************************************************************************************************************/
void deinit_uart(void)
{
    fsp_err_t err = FSP_SUCCESS;

    /* Close module */
#if (BSP_FEATURE_SCI_VERSION == 2U)
    err =  R_SCI_B_UART_Close (&g_uart9_ctrl);
#else
    err =  R_SCI_UART_Close (&g_uart9_ctrl);
#endif
}

/*****************************************************************************************************************
 *  @brief      UART user callback
 *  @param[in]  p_args
 *  @retval     None
 ****************************************************************************************************************/
void user_uart9_callback(uart_callback_args_t *p_args)
{
    /* Logged the event in global variable */
    g_uart_event = (uint8_t)p_args->event;

    /* Reset g_temp_buffer index if it exceeds than buffer size */
    if(DATA_LENGTH == g_counter_var)
    {
        g_counter_var = RESET_VALUE;
    }

		if(UART_EVENT_TX_COMPLETE == p_args->event )
    {
        uart_send_complete_flag = true;
    }   
		
    if(UART_EVENT_RX_CHAR == p_args->event)
    {
			  //send the char to the command line task
				cli_recv=p_args->data;
				xTaskNotifyFromISR(cli_command_task, (uint32_t)cli_recv, eSetValueWithOverwrite , NULL);
    }

 
		
		
}


int fputc(int ch, FILE *f)
{
		(void)f;
	
    fsp_err_t err = FSP_SUCCESS;
    err = R_SCI_UART_Write(&g_uart9_ctrl, (uint8_t *)&ch, 1);
    if(FSP_SUCCESS != err) __BKPT();
    while(uart_send_complete_flag == false){}
    uart_send_complete_flag = false;
    return ch;
}

/**
int _write(int file, char *data, int len)
{
		(void)file;
    fsp_err_t err = FSP_SUCCESS;
    err = R_SCI_UART_Write(&g_uart9_ctrl, (uint8_t*)data, (uint32_t)len);
    if(FSP_SUCCESS != err) __BKPT();
    while(uart_send_complete_flag == false){}
    uart_send_complete_flag = false;
    return len;
}
**/

/*******************************************************************************************************************//**
 * @} (end addtogroup r_sci_uart_ep)
 **********************************************************************************************************************/
