#include "hal_data.h"
#include "slcd.h"
#include "usr_rtc.h"
#include "uart_ep.h"
#include "common_utils.h"

FSP_CPP_HEADER
void R_BSP_WarmStart(bsp_warm_start_event_t event);
FSP_CPP_FOOTER

fsp_err_t update_rtc_time_info_to_lcd(void);

extern  rtc_time_t g_lcd_present_time;

volatile uint32_t g_periodic_irq_flag = RESET_FLAG;    /* flag to check occurrence of periodic interrupt */

static uint8_t  colon = RESET_VALUE;
uint8_t seg_value;

/*******************************************************************************************************************//**
 * main() is generated by the RA Configuration editor and is used to generate threads if an RTOS is used.  This function
 * is called by main() when no RTOS is used.
 **********************************************************************************************************************/
void hal_entry(void)
{
    /* TODO: add your own code here */
	  fsp_err_t err = FSP_SUCCESS;
    /* Initializing UART */
    err = uart_initialize();
    if (FSP_SUCCESS != err)
    {
        APP_ERR_TRAP(err);
    }	
	
    /* Initialize r_slcdc driver. */
    err = slcd_init ();
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT("\r\n ** SLCDC INITIALIZATION FAILED ** \r\n");
        APP_ERR_TRAP(err);
    }

    /* Set the Contrast level to Maximum */
    err = R_SLCDC_SetContrast (&g_slcdc_ctrl, SLCDC_CONTRAST_10);
    if (FSP_SUCCESS != err)
    {
        /* R_SLCDC_SetContrast failed */
        APP_ERR_PRINT("\r\n ** Setting the Contrast level failed ** \r\n");
        APP_ERR_TRAP(err);
    }
		
    /* Initialize RTC driver */
    err = rtc_init ();
    if (FSP_SUCCESS != err)
    {
        /* RTC module init failed */
        APP_ERR_PRINT("\r\n ** RTC INIT FAILED ** \r\n");
        APP_ERR_TRAP(err);
    }
    APP_PRINT("\r\nRTC  initialized and started");

    /* Sets the User Time to the RTC */
    err = set_rtc_calendar_time_with_user_configured_time ();
    if (FSP_SUCCESS != err)
    {
        /* set_rtc_calendar_time_with_user_configured_time failed */
        APP_ERR_PRINT("\r\n ** Setting the RTC Calendar Time with User Configured time : FAILED ** \r\n");
        APP_ERR_TRAP(err);
    }

    while(true)
    {				
        if(g_periodic_irq_flag)
        {
            err = update_rtc_time_info_to_lcd();
            if (FSP_SUCCESS != err)
            {
                /* update_rtc_time_info_to_lcd  failed */
                APP_ERR_PRINT("\r\n ** Updating RTC time to LCD  : FAILED ** \r\n");
                APP_ERR_TRAP(err);
            }

            g_periodic_irq_flag = RESET_VALUE;
				}
    }		
#if BSP_TZ_SECURE_BUILD
    /* Enter non-secure code */
    R_BSP_NonSecureEnter();
#endif
}

/*******************************************************************************************************************//**
 * @brief       This functions updates the RTC time to the LCD. .
 * @param[IN]   None
 * @retval      None
 * @retval      Any Other Error code apart from FSP_SUCCESS for Unsuccessful getting calendar time, setting the segment.
 **********************************************************************************************************************/
fsp_err_t update_rtc_time_info_to_lcd(void)
{
    fsp_err_t err = FSP_SUCCESS;
    uint32_t time_value_hr = RESET_VALUE;
    uint32_t time_value_mn = RESET_VALUE;
		uint32_t time_value_s = RESET_VALUE;

    err = get_rtc_calendar_time ();
    if (FSP_SUCCESS != err)
    {
        /* RTC Calendar time read  failed */
        APP_ERR_PRINT("\r\n ** RTC READ FAILED ** \r\n");
        APP_ERR_TRAP(err);
    }
		
    time_value_hr = (uint32_t) (g_lcd_present_time.tm_hour);
    time_value_mn = (uint32_t) (g_lcd_present_time.tm_min);
		time_value_s	=	(uint32_t) (g_lcd_present_time.tm_sec);

   /* Concatenate the Minutes and Seconds for the LCD Display purpose*/
    time_value_hr = (time_value_hr * 10000) + (uint32_t) (time_value_mn*100)+time_value_s;

		err = set_segments_medium_digits (time_value_hr, SET_COLON);
		err = set_segments_medium_digits (time_value_hr, SET_DP);
		
		return(err);
}


/*******************************************************************************************************************//**
 * This function is called at various points during the startup process.  This implementation uses the event that is
 * called right before main() to set up the pins.
 *
 * @param[in]  event    Where at in the start up process the code is currently at
 **********************************************************************************************************************/
void R_BSP_WarmStart (bsp_warm_start_event_t event)
{
    if (BSP_WARM_START_RESET == event)
    {
#if BSP_FEATURE_FLASH_LP_VERSION != 0

        /* Enable reading from data flash. */
        R_FACI_LP->DFLCTL = 1U;

        /* Would normally have to wait tDSTOP(6us) for data flash recovery. Placing the enable here, before clock and
         * C runtime initialization, should negate the need for a delay since the initialization will typically take more than 6us. */
#endif
    }

    if (BSP_WARM_START_POST_C == event)
    {
        /* C runtime environment and system clocks are setup. */

        /* Configure pins. */
        R_IOPORT_Open(&IOPORT_CFG_CTRL, &IOPORT_CFG_NAME);

#if BSP_CFG_SDRAM_ENABLED

        /* Setup SDRAM and initialize it. Must configure pins first. */
        R_BSP_SdramInit(true);
#endif
    }
}

#if BSP_TZ_SECURE_BUILD

FSP_CPP_HEADER
BSP_CMSE_NONSECURE_ENTRY void template_nonsecure_callable ();

/* Trustzone Secure Projects require at least one nonsecure callable function in order to build (Remove this if it is not required to build). */
BSP_CMSE_NONSECURE_ENTRY void template_nonsecure_callable ()
{

}
FSP_CPP_FOOTER

#endif
