
#include "slcd.h"
#include "hal_data.h"
#include "common_utils.h"

#define CLEAR_WRITE_DATA    (0x00)

const uint8_t medium_digit_start_addresses[6][2]=
{
    { 0x29,  0x28 },  // units  digit is controlled by SLCD Data register 41 & 40
    { 0x27,  0x1E },  // tens digit is controlled by SLCD Data register 39 & 30
    { 0x1D,  0x18 },  // hundreds digit is controlled by SLCD Data register 29 & 24
    { 0x17,  0x16 },  // thousands digit is controlled by SLCD Data register 23 & 22
    { 0x10,  0x0F },  // ten-thousands digit is controlled by SLCD Data register 16 & 15
    { 0x0B,  0x03 }   // hundred-thousands digit is controlled by SLCD Data register 11 & 3
};

const uint8_t medium_digit_data[10][2]=
{
    { 0x07, 0x0D }, // 0 digit is display by writing 0x0f and 0x0a to the appropriate medium_digit_start_addresses
    { 0x06, 0x00 }, // 1 digit is display by writing 0x06 and 0x00 to the appropriate medium_digit_start_addresses
    { 0x03, 0x0E }, // 2 digit is display by writing 0x0d and 0x06 to the appropriate medium_digit_start_addresses
    { 0x07, 0x0A }, // 3 digit is display by writing 0x0f and 0x04 to the appropriate medium_digit_start_addresses
    { 0x06, 0x03 }, // 4 digit is display by writing 0x06 and 0x0c to the appropriate medium_digit_start_addresses
    { 0x05, 0x0B }, // 5 digit is display by writing 0x0b and 0x0c to the appropriate medium_digit_start_addresses
    { 0x05, 0x0F }, // 6 digit is display by writing 0x0b and 0x0e to the appropriate medium_digit_start_addresses
    { 0x07, 0x00 }, // 7 digit is display by writing 0x0e and 0x00 to the appropriate medium_digit_start_addresses
    { 0x07, 0x0F }, // 8 digit is display by writing 0x0f and 0x0e to the appropriate medium_digit_start_addresses
    { 0x07, 0x0B }  // 9 digit is display by writing 0x0f and 0x0c to the appropriate medium_digit_start_addresses
};

const uint8_t medium_digit_error_data[5][2]=
{
    { 0x01, 0x0F }, // E character is display by writing 0x09 and 0x0e to the appropriate medium_digit_start_addresses
    { 0x01, 0x05 }, // r character is display by writing 0x00 and 0x06 to the appropriate medium_digit_start_addresses
    { 0x01, 0x05 }, // r character is display by writing 0x03 and 0x06 to the appropriate medium_digit_start_addresses
    { 0x04, 0x0E }, // o character is display by writing 0x03 and 0x06 to the appropriate medium_digit_start_addresses
    { 0x01, 0x05 }  // r character is display by writing 0x03 and 0x06 to the appropriate medium_digit_start_addresses
};

/* This Array holds the Data being updated to individual Segments for the Application */
static volatile  uint8_t running_segment_data[10][2]=
{
    { 0x00,  0x00 },  // stores running segment data for SEG11(ABC) & 3(DEGF) represent from right to left 6th number
    { 0x00,  0x00 },  // stores running segment data for SEG16(ABC-COL1) & 15(DEGF) represent from right to left 5th number and COM3 Control COL1
    { 0x00,  0x00 },  // stores running segment data for SEG23(ABC) & 22(DEGF) represent from right to left 4th number
    { 0x00,  0x00 },  // stores running segment data for SEG29(ABC-DOT1) & 24(DEGF) represent from right to left 3th number
    { 0x00,  0x00 },  // stores running segment data for SEG39(ABC) & 30(DEGF) represent from right to left 2th number
    { 0x00,  0x00 },  // stores running segment data for SEG41(ABC) & 40(DEGF) represent from right to left 1th number
};


const icon_segment_t battery_level_segment[4] =
{
    { BATTERY_SHELL_SEGMENT,    BATTERY_SHELL_DATA  },                                                                      // BATTERY_EMPTY OR BATTERY_0
    { BATTERY_LEVEL1_SEGMENT,   BATTERY_LEVEL1_DATA },                                                                      // BATTERY_1
    { BATTERY_LEVEL2_SEGMENT, ( BATTERY_LEVEL1_DATA | BATTERY_LEVEL2_DATA ) },                                              // BATTERY_2
    { BATTERY_LEVEL3_SEGMENT, ( BATTERY_LEVEL1_DATA | BATTERY_LEVEL2_DATA | BATTERY_LEVEL3_DATA ) }                         // BATTERY_3
};

/*******************************************************************************************************************//**
 *  @brief       Performs writing Segment LCD's medium Digits with user requested value and DP settings .
 *               This is used for displaying the time information in this application along with colon icon
 *  @param[IN]   None
 *  @retval      FSP_SUCCESS     Upon successful writing.
 *  @retval      Any Other Error code apart from FSP_SUCCESS on Unsuccessful operation .
 **********************************************************************************************************************/
fsp_err_t set_segments_medium_digits( uint32_t value, medium_dp_or_colon_t point_or_colon )
{
		uint8_t	temp_hund_thousands= (uint8_t)((value / 100000));
    uint8_t temp_ten_thousands = (uint8_t)((value % 100000  / 10000));
    uint8_t temp_thousands     = (uint8_t)((value % 10000) / 1000);
    uint8_t temp_hundreds      = (uint8_t)((value % 1000) / 100);
    uint8_t temp_tens          = (uint8_t)((value % 100) / 10);
    uint8_t temp_units         = (uint8_t)((value % 10) / 1);

    fsp_err_t err;
    // set / clear the decimal point
    err = set_segments_icon_medium_dp_colon( point_or_colon );

    if( value > 999999)
    {
        // Write an error message "E r r o r"
        err = set_segments_icon_medium_dp_colon(  DP_COLON_OFF );
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TEN_THOUSANDS][0], &medium_digit_error_data[0][0], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TEN_THOUSANDS][1], &medium_digit_error_data[0][1], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);

        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][0], &medium_digit_error_data[1][0], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][1], &medium_digit_error_data[1][1], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);

        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][0], &medium_digit_error_data[2][0], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][1], &medium_digit_error_data[2][1], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);

        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][0], &medium_digit_error_data[3][0], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][1], &medium_digit_error_data[3][1], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);

        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_UNITS][0], &medium_digit_error_data[4][0], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Write( &g_slcdc_ctrl, medium_digit_start_addresses[MN_UNITS][1], &medium_digit_error_data[4][1], 1);
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    else
    {
				if (value > 99999)
				{
            running_segment_data[MN_HUN_THOUSANDS][0] = running_segment_data[MN_HUN_THOUSANDS][0] & MED_DIGIT_MASK_4_2_1;
            running_segment_data[MN_HUN_THOUSANDS][1] = running_segment_data[MN_HUN_THOUSANDS][1] & MED_DIGIT_MASK_8_4_2_1;

            running_segment_data[MN_HUN_THOUSANDS][0] = (running_segment_data[MN_HUN_THOUSANDS][0] | medium_digit_data[temp_hund_thousands][0]);
            running_segment_data[MN_HUN_THOUSANDS][1] = (running_segment_data[MN_HUN_THOUSANDS][1] | medium_digit_data[temp_hund_thousands][1]);
            err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)medium_digit_start_addresses[MN_HUN_THOUSANDS][0], (uint8_t)running_segment_data[MN_HUN_THOUSANDS][0], (uint8_t)~(uint8_t)MED_DIGIT_MASK_4_2_1);
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, (uint8_t)medium_digit_start_addresses[MN_HUN_THOUSANDS][1], (uint8_t)running_segment_data[MN_HUN_THOUSANDS][1], (uint8_t)~(uint8_t)MED_DIGIT_MASK_8_4_2_1);
            CHECK_SEGMENT_WRITE_ERROR(err);
				}
				else
				{
            running_segment_data[MN_HUN_THOUSANDS][0] = (running_segment_data[MN_HUN_THOUSANDS][0] | CLEAR_WRITE_DATA);
            running_segment_data[MN_HUN_THOUSANDS][1] = (running_segment_data[MN_HUN_THOUSANDS][1] | CLEAR_WRITE_DATA);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUN_THOUSANDS][0], running_segment_data[MN_HUN_THOUSANDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUN_THOUSANDS][1], running_segment_data[MN_HUN_THOUSANDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);					
				}
        if( value > 9999 )
        {
            running_segment_data[MN_TEN_THOUSANDS][0] = running_segment_data[MN_TEN_THOUSANDS][0] & MED_DIGIT_MASK_4_2_1;
            running_segment_data[MN_TEN_THOUSANDS][1] = running_segment_data[MN_TEN_THOUSANDS][1] & MED_DIGIT_MASK_8_4_2_1;

            running_segment_data[MN_TEN_THOUSANDS][0] = (running_segment_data[MN_TEN_THOUSANDS][0] | medium_digit_data[temp_ten_thousands][0]);
            running_segment_data[MN_TEN_THOUSANDS][1] = (running_segment_data[MN_TEN_THOUSANDS][1] | medium_digit_data[temp_ten_thousands][1]);
            err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)medium_digit_start_addresses[MN_TEN_THOUSANDS][0], (uint8_t)running_segment_data[MN_TEN_THOUSANDS][0], (uint8_t)~(uint8_t)MED_DIGIT_MASK_4_2_1);
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, (uint8_t)medium_digit_start_addresses[MN_TEN_THOUSANDS][1], (uint8_t)running_segment_data[MN_TEN_THOUSANDS][1], (uint8_t)~(uint8_t)MED_DIGIT_MASK_8_4_2_1);
            CHECK_SEGMENT_WRITE_ERROR(err);
        }
        else
        {
            running_segment_data[MN_TEN_THOUSANDS][0] = (running_segment_data[MN_TEN_THOUSANDS][0] | CLEAR_WRITE_DATA);
            running_segment_data[MN_TEN_THOUSANDS][1] = (running_segment_data[MN_TEN_THOUSANDS][1] | CLEAR_WRITE_DATA);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TEN_THOUSANDS][0], running_segment_data[MN_TEN_THOUSANDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TEN_THOUSANDS][1], running_segment_data[MN_TEN_THOUSANDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }

        if( value > 999 )
        {
            running_segment_data[MN_THOUSANDS][0] = running_segment_data[MN_THOUSANDS][0] & MED_DIGIT_MASK_4_2_1;
            running_segment_data[MN_THOUSANDS][1] = running_segment_data[MN_THOUSANDS][1] & MED_DIGIT_MASK_8_4_2_1;

            running_segment_data[MN_THOUSANDS][0] = (running_segment_data[MN_THOUSANDS][0] | medium_digit_data[temp_thousands][0]);
            running_segment_data[MN_THOUSANDS][1] = (running_segment_data[MN_THOUSANDS][1] | medium_digit_data[temp_thousands][1]);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][0], running_segment_data[MN_THOUSANDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][1], running_segment_data[MN_THOUSANDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }
        else
        {
            running_segment_data[MN_THOUSANDS][0] = (running_segment_data[MN_THOUSANDS][0] | CLEAR_WRITE_DATA);
            running_segment_data[MN_THOUSANDS][1] = (running_segment_data[MN_THOUSANDS][1] | CLEAR_WRITE_DATA);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][0],  running_segment_data[MN_THOUSANDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_THOUSANDS][1],  running_segment_data[MN_THOUSANDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }

        if( value > 99 )
        {
            running_segment_data[MN_HUNDREDS][0] = running_segment_data[MN_HUNDREDS][0] & MED_DIGIT_MASK_4_2_1;
            running_segment_data[MN_HUNDREDS][1] = running_segment_data[MN_HUNDREDS][1] & MED_DIGIT_MASK_8_4_2_1;

            running_segment_data[MN_HUNDREDS][0] = (running_segment_data[MN_HUNDREDS][0] | medium_digit_data[temp_hundreds][0]);
            running_segment_data[MN_HUNDREDS][1] = (running_segment_data[MN_HUNDREDS][1] | medium_digit_data[temp_hundreds][1]);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][0], running_segment_data[MN_HUNDREDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][1], running_segment_data[MN_HUNDREDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }
        else
        {
            running_segment_data[MN_HUNDREDS][0] = (running_segment_data[MN_HUNDREDS][0] | CLEAR_WRITE_DATA);
            running_segment_data[MN_HUNDREDS][1] = (running_segment_data[MN_HUNDREDS][1] | CLEAR_WRITE_DATA);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][0], running_segment_data[MN_HUNDREDS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_HUNDREDS][1], running_segment_data[MN_HUNDREDS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }

        if( value > 9 )
        {
            running_segment_data[MN_TENS][0] = running_segment_data[MN_TENS][0] & MED_DIGIT_MASK_4_2_1;
            running_segment_data[MN_TENS][1] = running_segment_data[MN_TENS][1] & MED_DIGIT_MASK_8_4_2_1;

            running_segment_data[MN_TENS][0] = (running_segment_data[MN_TENS][0] | medium_digit_data[temp_tens][0]);
            running_segment_data[MN_TENS][1] = (running_segment_data[MN_TENS][1] | medium_digit_data[temp_tens][1]);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][0], running_segment_data[MN_TENS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][1], running_segment_data[MN_TENS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }
        else
        {
            running_segment_data[MN_TENS][0] = (running_segment_data[MN_TENS][0] | CLEAR_WRITE_DATA);
            running_segment_data[MN_TENS][1] = (running_segment_data[MN_TENS][1] | CLEAR_WRITE_DATA);

            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][0], running_segment_data[MN_TENS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
            err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_TENS][1], running_segment_data[MN_TENS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
            CHECK_SEGMENT_WRITE_ERROR(err);
        }

        running_segment_data[MN_UNITS][0] = running_segment_data[MN_UNITS][0] & MED_DIGIT_MASK_4_2_1;
        running_segment_data[MN_UNITS][1] = running_segment_data[MN_UNITS][1] & MED_DIGIT_MASK_8_4_2_1;
        // Write the Units
        running_segment_data[MN_UNITS][0] = (running_segment_data[MN_UNITS][0] | medium_digit_data[temp_units][0]);
        running_segment_data[MN_UNITS][1] = (running_segment_data[MN_UNITS][1] | medium_digit_data[temp_units][1]);

        err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_UNITS][0], running_segment_data[MN_UNITS][0], (uint8_t)~MED_DIGIT_MASK_4_2_1 );
        CHECK_SEGMENT_WRITE_ERROR(err);
        err = R_SLCDC_Modify( &g_slcdc_ctrl, medium_digit_start_addresses[MN_UNITS][1], running_segment_data[MN_UNITS][1], (uint8_t)~MED_DIGIT_MASK_8_4_2_1 );
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    return err;
}


/*******************************************************************************************************************//**
 *  @brief       Performs writing Segment LCD's bATTERY ICON to  set/clear display.
 *  @param[IN]   None
 *  @retval      FSP_SUCCESS     Upon successful writing.
 *  @retval      Any Other Error code apart from FSP_SUCCESS on Unsuccessful operation .
 **********************************************************************************************************************/
fsp_err_t set_segments_icon_battery( battery_level_t level )
{
    fsp_err_t err;

    if( BATTERY_OFF == level )
    {
        err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)BATTERY_SHELL_SEGMENT, (uint8_t)CLEAR_WRITE_DATA, (uint8_t)BATTERY_SHELL_DATA);
        err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)BATTERY_LEVEL1_SEGMENT, (uint8_t)CLEAR_WRITE_DATA, (uint8_t)( BATTERY_LEVEL1_DATA | BATTERY_LEVEL2_DATA | BATTERY_LEVEL3_DATA) );
    }
    else
    {
        // Always draw the outer battery symbol
        err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)battery_level_segment[0].segment, (uint8_t)battery_level_segment[0].data, (uint8_t) ~battery_level_segment[0].data );

        // Clear all the existing level bars
        err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)battery_level_segment[level].segment, (uint8_t)CLEAR_WRITE_DATA, (uint8_t) ~battery_level_segment[level].data );

        // Write the new level bars
        err = R_SLCDC_Modify(&g_slcdc_ctrl, (uint8_t)battery_level_segment[level].segment, (uint8_t)battery_level_segment[level].data, (uint8_t) ~battery_level_segment[level].data );
    }
    return err;
}

/*******************************************************************************************************************//**
 *  @brief       Performs writing Segment LCD's dp AND colon to  set/clear display the specified ICONS.
 *  @param[IN]   None
 *  @retval      FSP_SUCCESS     Upon successful writing.
 *  @retval      Any Other Error code apart from FSP_SUCCESS on Unsuccessful operation .
 **********************************************************************************************************************/
fsp_err_t set_segments_icon_medium_dp_colon( medium_dp_or_colon_t setting )
{
    fsp_err_t err = FSP_SUCCESS;
    if(DP_COLON_OFF == setting)
    {
        // Clear DECIMAL POINT & COLON
        running_segment_data[3][0] = (running_segment_data[3][0] & (uint8_t)(~(MEDIUM_DIGIT_COLON_DATA|MEDIUM_DIGIT_DEC_POINT_DATA)));
        err = R_SLCDC_Modify(&g_slcdc_ctrl, MEDIUM_DIGIT_COLON_SEGMENT, (uint8_t)running_segment_data[3][0], (MEDIUM_DIGIT_COLON_DATA|MEDIUM_DIGIT_DEC_POINT_DATA) );
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    if(COLON_OFF == setting)
    {
        running_segment_data[1][0] = (running_segment_data[1][0] & (uint8_t)(~MEDIUM_DIGIT_COLON_DATA));
        // Clear DECIMAL POINT & COLON
        err = R_SLCDC_Modify(&g_slcdc_ctrl, MEDIUM_DIGIT_COLON_SEGMENT, (uint8_t)running_segment_data[1][0], MEDIUM_DIGIT_COLON_DATA);
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    else if( SET_COLON == setting)
    {
        running_segment_data[3][0] = (running_segment_data[3][0] | MEDIUM_DIGIT_COLON_DATA);
        // Write the specified segment
        err = R_SLCDC_Modify(&g_slcdc_ctrl, MEDIUM_DIGIT_COLON_SEGMENT, running_segment_data[3][0], (uint8_t) MEDIUM_DIGIT_COLON_DATA );
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    else if ( SET_DP == setting)
    {
        running_segment_data[1][0] = (running_segment_data[1][0] | MEDIUM_DIGIT_DEC_POINT_DATA);
        // Write the specified segment
        err = R_SLCDC_Modify(&g_slcdc_ctrl, MEDIUM_DIGIT_DEC_POINT_SEGMENT, (uint8_t)running_segment_data[1][0], (uint8_t) MEDIUM_DIGIT_DEC_POINT_DATA);
        CHECK_SEGMENT_WRITE_ERROR(err);
    }
    return err;
}

/*******************************************************************************************************************//**
 *  @brief       Performs open, start and set display area operations.
 *  @param[IN]   None
 *  @retval      FSP_SUCCESS     Upon successful SLCDC Initialization.
 *  @retval      Any Other Error code apart from FSP_SUCCESS on Unsuccessful operation .
 **********************************************************************************************************************/
fsp_err_t slcd_init(void)
{
    fsp_err_t err = FSP_SUCCESS;

    /* Open SLCDC driver.*/
    err = R_SLCDC_Open(&g_slcdc_ctrl, &g_slcdc_cfg);
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\n ** R_SLCDC_Open API FAILED ** \r\n");
        return err;
    }

    R_BSP_SoftwareDelay (LCD_STABILIZATION_DELAY, BSP_DELAY_UNITS_MILLISECONDS);

    /* Start SLCDC driver.*/
    err = R_SLCDC_Start(&g_slcdc_ctrl);
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\n ** R_SLCDC_Start API FAILED ** \r\n");
        /* de-initialize the opened slcdc module before trap*/
        deinit_slcdc();
        return err;
    }

    /* Set Display Area of SLCDC driver.*/
    err = R_SLCDC_SetDisplayArea(&g_slcdc_ctrl, SLCDC_DISP_A);
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\n ** R_SLCDC_SetDisplayArea API FAILED ** \r\n");
        /* de-initialize the opened slcdc module before trap*/
        deinit_slcdc();
    }
    return err;
}


/*****************************************************************************************************************
 * @brief      Close the SLCDC HAL driver.
 * @param[IN]  None
 * @retval     None
 ****************************************************************************************************************/
void deinit_slcdc(void)
{
    fsp_err_t err = FSP_SUCCESS;

    /* Close slcdc module */
    err = R_SLCDC_Close(&g_slcdc_ctrl);
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\n** R_SLCDC_Close API FAILED ** \r\n");
    }
}


