/***********************************************************************************************************************
 * File Name    : usr_rtc.c
 * Description  : Contains data structures and functions used in usr_rtc.c.
 **********************************************************************************************************************/
/***********************************************************************************************************************
* Copyright (c) 2020 - 2024 Renesas Electronics Corporation and/or its affiliates
*
* SPDX-License-Identifier: BSD-3-Clause
***********************************************************************************************************************/

#include <usr_rtc.h>
#include "common_utils.h"


/*******************************************************************************************************************//**
 * @addtogroup rtc_ep
 * @{
 **********************************************************************************************************************/

/* Variable to store time */

rtc_time_t g_set_time =
{
 .tm_sec     =  USER_DATA_SECONDS,                 /* seconds  - [0, 59] */
 .tm_min     =  USER_DATA_MINUTES,                 /* minutes  - [0, 59] */
 .tm_hour    =  USER_DATA_HOUR,                    /* hours  - [0, 23] */
 .tm_mday    =  USER_DATA_DAY_OF_THE_MONTH,        /* day of the month - [1, 31] */
 .tm_mon     =  USER_DATA_MONTH_OF_THE_YEAR,       /* months January = 0, Feb = 1,  [0, 11] */
 .tm_year    =  USER_DATA_YEAR,                    /* years since 1900 */
};

static rtc_time_t g_present_time ;
rtc_time_t g_lcd_present_time ;
static uint32_t set_time_flag = RESET_FLAG;  //Flag to be set on successful RTC Calendar setting

extern volatile uint32_t g_periodic_irq_flag;


/*******************************************************************************************************************//**
 * @brief       This functions initializes RTC module.
 * @param[IN]   None
 * @retval      FSP_SUCCESS                  Upon successful open of RTC module
 * @retval      Any Other Error code apart from FSP_SUCCESS  Unsuccessful open
 **********************************************************************************************************************/
fsp_err_t rtc_init(void)
{
    fsp_err_t err = FSP_SUCCESS;     // Error status

    /* Open RTC module */
    err = R_RTC_Open(&g_rtc_ctrl, &g_rtc_cfg);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\nRTC module open failed.\r\nRestart the Application\r\n");
    }

    err = R_RTC_PeriodicIrqRateSet (&g_rtc_ctrl, RTC_PERIODIC_IRQ_SELECT_1_SECOND);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT ("\r\nRTC Periodic Irq set  failed.\r\nRestart the Application\r\n");
    }

    return err;
}


/*******************************************************************************************************************//**
 * @brief       This functions sets the Calendar time provided by user.
 * @param[IN]   None
 * @retval      FSP_SUCCESS                  Upon successful Calendar time set
 * @retval      Any Other Error code apart from FSP_SUCCESS  Unsuccessful Calendar time set
 **********************************************************************************************************************/
fsp_err_t set_rtc_calendar_time_with_user_configured_time(void)
{
    fsp_err_t err = FSP_SUCCESS;     // Error status

    APP_PRINT ("\r\nSetting RTC Date and Time \r\n");

    err = R_RTC_CalendarTimeSet(&g_rtc_ctrl, &g_set_time);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT("\r\nCalendarTime Set failed.\r\n");
        return err;
    }

    /* Get the current Calendar time */
    err = R_RTC_CalendarTimeGet(&g_rtc_ctrl, &g_present_time);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT("\r\nCalendarTime Get failed.\r\n");
        return err;
    }
    /* Modify the date in standard format to user readable format */
    rtc_date_readability_update(&g_present_time);

    g_lcd_present_time = g_present_time;

    APP_PRINT("\r\n RTC calendar set to  Date : %2d/%2d/%4d \n Time  : %2d : %2d : %2d \r\n\n", g_present_time.tm_mday,
            g_present_time.tm_mon, g_present_time.tm_year, g_present_time.tm_hour,
            g_present_time.tm_min, g_present_time.tm_sec);

    set_time_flag = SET_FLAG;

    return err;
}


/*******************************************************************************************************************//**
 * @brief       This functions gets the current RTC time.
 * @param[IN]   None
 * @retval      FSP_SUCCESS                  Upon successful current RTC time get
 * @retval      Any Other Error code apart from FSP_SUCCESS  Unsuccessful current RTC time get
 **********************************************************************************************************************/
fsp_err_t get_rtc_calendar_time(void)
{
    fsp_err_t err = FSP_SUCCESS;     // Error status

    /* Get the current RTC Calendar time */
    err = R_RTC_CalendarTimeGet(&g_rtc_ctrl, &g_present_time);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        APP_ERR_PRINT("\r\nGetting RTC Calendar time failed.\r\n");
        return err;
    }

    /* Modify the date in standard format to user readable format */
    rtc_date_readability_update(&g_present_time);

    g_lcd_present_time = g_present_time;

    APP_PRINT("\r\n RTC  Date : %d/%d/%d     Time : %d : %d : %d   ", g_present_time.tm_mday,
            g_present_time.tm_mon , g_present_time.tm_year, g_present_time.tm_hour,
            g_present_time.tm_min, g_present_time.tm_sec);

    return err;
}

/*******************************************************************************************************************//**
 * @brief       This functions modifies the date in readable format to the user.
 * @param[IN]   time                    date to be modified
 * @retval      None
 **********************************************************************************************************************/
void rtc_date_readability_update(rtc_time_t * time)
{
    time->tm_mon  +=  MON_ADJUST_VALUE;
    time->tm_year +=  YEAR_ADJUST_VALUE;
}

/*******************************************************************************************************************//**
 * @brief RTC callback function.
 * @param[in]  p_args
 * @retval     None
 **********************************************************************************************************************/
void rtc_callback(rtc_callback_args_t *p_args)
{
    if(RTC_EVENT_PERIODIC_IRQ == p_args->event)
    {
        g_periodic_irq_flag = SET_FLAG;
    }
    else
    {
        g_periodic_irq_flag = RESET_FLAG;
    }
}

/*******************************************************************************************************************//**
 * @brief This function closes opened RTC module before the project ends up in an Error Trap.
 * @param[IN]   None
 * @retval      None
 **********************************************************************************************************************/
void rtc_deinit(void)
{
    fsp_err_t err = FSP_SUCCESS;

    /* Close RTC module */
    err = R_RTC_Close(&g_rtc_ctrl);

    /* Handle error */
    if (FSP_SUCCESS != err)
    {
        /* RTC Close failure message */
        APP_ERR_PRINT("** RTC module Close failed **  \r\n");
    }
}

/*******************************************************************************************************************//**
 * @} (end addtogroup usr_rtc)
 **********************************************************************************************************************/

